import { NextRequest, NextResponse } from 'next/server';
import Database from 'better-sqlite3';
import path from 'path';

function getDb() {
    const dbPath = path.join(process.cwd(), 'domains.db');
    return new Database(dbPath);
}

// GET /api/events/[id]/tasks - קבלת משימות של אירוע
export async function GET(request: NextRequest, { params }: { params: Promise<{ id: string }> }) {
    const { id } = await params;
    const db = getDb();
    try {
        // בדיקה שהאירוע קיים
        const event = db.prepare('SELECT id, title FROM events WHERE id = ?').get(id) as unknown;
        if (!event) {
            return NextResponse.json({ error: 'אירוע לא נמצא' }, { status: 404 });
        }

        // קבלת משימות קשורות
        const tasks = db.prepare(`
            SELECT 
                t.*,
                c.name as customer_name,
                c.company as customer_company,
                d.domain as domain_name,
                p.title as project_title
            FROM tasks t
            LEFT JOIN customers c ON t.customer_id = c.id
            LEFT JOIN domains d ON t.domain_id = d.id
            LEFT JOIN projects p ON t.project_id = p.id
            WHERE t.event_id = ?
            ORDER BY t.due_date ASC, t.priority DESC
        `).all(id) as unknown as any[];

        return NextResponse.json({
            event: event,
            tasks: tasks,
            count: tasks.length
        });

    } catch (e: unknown) {
        console.error('Get event tasks error:', e);
        return NextResponse.json({ error: 'שגיאה בטעינת משימות אירוע: ' + (e as Error).message }, { status: 500 });
    } finally {
        db.close();
    }
}
